function [ q, iter, qTol, ELBO, A ] = qSolver( q0, H, y, Pn, qTol, maxIters )
%qSolver Summary of this function goes here
% This function performs fixed point iterations to solve the system of
% non-linear equations that must be satisfied by the true soft-symbol
% vector q.
%
% q0: initial soft symbol vector
% H: NLx2N real valued channel matrix
% y: NLx1 vector of measurements (L>=2, to satisfy Nyquist sampling)
% Pn: Noise power
% qTol: tolerance (on change in q) for stopping fp itertaions
% maxIters: maximum number of fixed point iterations
%
%References
%   1) Arunkumar K.P. and Chandra R. Murthy, "Variational Soft Symbol
%   Decoding for Sweep Spread Carrier Based Underwater Acoustic
%   Communication", SPAWC 2019, Cannes, France.
%
%   2) Arunkumar K. P. and C. R. Murthy, Soft Symbol Decoding in
%   Sweep-Spread-Carrier Underwater Acoustic Communications: A Novel
%   Variational Bayesian Algorithm and its Analysis, Accepted, IEEE
%   Transactions on Signal Processing, Mar. 2020.
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%Revision History
% Version : 1.0
% Last Revision: 05-10-2019
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
%
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

q = q0; %intialize q

ELBO = zeros(maxIters+1,1); %evidence lower bound

A = zeros(maxIters+1,1); %direction cosine of fixed point update direction wrt ELBO gradient

nu = 1/sqrt(2)*bsxfun(@times, H,  2*q'-1);

eta = bsxfun(@times, H(:,1:2:end).*H(:,2:2:end),...
    (2*q(1:2:end)'-1).*(2*q(2:2:end)'-1));

ELBO(1) = 1/(sqrt(2)*Pn)*y'*(H*(2*q-1))...
    - 1/(2*Pn)*( sum(sum(eta)) ...
    + sum(sum(nu.*bsxfun(@minus, sum(nu,2), nu))) ); %initial value of lower bound at q = q0

q_ = q; %q_ is used to save the q in the previous iteration

q_(q_<qTol/100) = qTol/100; %avoiding q=0 so that log(q)>-inf

q_(q_>1-qTol/100) = 1 - qTol/100; %avoiding q=1 so that log(1-q)<inf

u = sqrt(2)/Pn*( y'*H - sum( H.*bsxfun(@minus, sum(nu,2), nu) ) );

qTemp = 1./(1+exp(-u.'));

dL = log(1./q_ - 1) + u(:);

A(1) = (qTemp-q_)'*dL/(norm(qTemp-q_,2)*norm(dL,2));

alpha = [0 0.0001 0.001 0.01 0.05 0.1 0.2 0.3 0.4 0.5 1.0 2 4 8 16];%step size candidate values

Ltest = -inf*ones(length(alpha),1);

for iter = 1:maxIters
    q_ = q;  %q in the previous iteration
    
    nu = 1/sqrt(2)*bsxfun(@times, H,  2*q'-1);
    
    u = sqrt(2)/Pn*( y'*H - sum( H.*bsxfun(@minus, sum(nu,2), nu) ) );
    
    qTemp = 1./(1+exp(-u.'));
    
    %Determine the ELBO for different candidate step sizes used for
    %updating q along qTemp - q_
    for iAlpha = 1:length(alpha)
        q = q_ + alpha(iAlpha)*(qTemp - q_);
        
        q(q<0) = 0; %avoiding q < 0
        
        q(q>1) = 1; %avoiding q > 0
        
        if (min(q)>=0) && (max(q)<=1)
            eta = bsxfun(@times, H(:,1:2:end).*H(:,2:2:end),...
                (2*q(1:2:end)'-1).*(2*q(2:2:end)'-1));
            
            nu = 1/sqrt(2)*bsxfun(@times, H,  2*q'-1);
            
            Ltest(iAlpha) = 1/(sqrt(2)*Pn)*y'*(H*(2*q-1))...
                - 1/(2*Pn)*( sum(sum(eta))...
                + sum(sum(nu.*bsxfun(@minus, sum(nu,2), nu))) );
        else
            Ltest(iAlpha) = -inf;
        end
    end
    
    %Update q along (qTemp - q_) by an amount that best increases
    %ELBO
    if max(Ltest)>ELBO(iter)%update q only if the current ELBO value is better than the previous
        [ELBO(iter+1), iMax] = max(Ltest);
        %                 disp(num2str(iMax));
        q = q_ + alpha(iMax)*(qTemp - q_);
        
        q(q<0) = 0; %avoiding q < 0
        
        q(q>1) = 1; %avoiding q > 0
    else%maintain q at its previous value if the current ELBO value is NOT better than the previous
        ELBO(iter+1) = ELBO(iter);
        
        q = q_;
    end
    
    q_(q_<qTol/100) = qTol/100; %avoiding q=0 so that log(q)>-inf
    
    q_(q_>1-qTol/100) = 1 - qTol/100; %avoiding q=1 so that log(1-q)<inf
    
    dL = log(1./q_ - 1) + u(:);
    
    A(iter+1) = (qTemp-q_)'*dL/(norm(qTemp-q_,2)*norm(dL,2)); %direction cosine of fixed point update direction wrt ELBO gradient
        
    dQmag = norm(q-q_,2);%change in q from last iteration
    
    if (ELBO(iter+1)-ELBO(iter) < 1e-3*ELBO(iter)) && (dQmag<qTol)
        %         disp(iter);
        break;%stop iterating if neither the ELBO nor q has changed significantly
    end
    
    if iter>10 && (mean(diff(ELBO(iter-10:iter))) < 1e-3*ELBO(iter))
        %         disp(iter);
        break;%stop iterating if the ELBO has been approximately maintaining its value over last 10 (consecutive) iterations
    end
        
end

ELBO = ELBO(1:iter+1);

A = A(1:iter+1);

end